<?php
session_start();
include("connect.php"); // provides $con (mysqli)
include("env-helper.php");

// ----- Security headers -----
header('X-Frame-Options: SAMEORIGIN');
header('X-Content-Type-Options: nosniff');

// ----- Auth check -----
if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("Location: login.php");
    exit;
}

// ----- CSRF token -----
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$CSRF_TOKEN = $_SESSION['csrf_token'];

// ----- Load Paystack secret for withdrawal -----
$paystack_secret = getEnvValue('sk_live_73d65edb013cfd44efd4410d648951bc8fafd176'); // matches your .env
if (!$paystack_secret) {
    http_response_code(500);
    die('Server not configured. (PAYSTACK_WITHDRAW_SECRET missing)');
}

// ----- Utilities -----
function log_event($msg) {
    @file_put_contents(__DIR__ . '/paystack_withdraw.log', "[" . date('c') . "] " . $msg . PHP_EOL, FILE_APPEND);
}

// Centralized Paystack caller function (unchanged)
function paystack_call($endpoint, $post = null) {
    global $paystack_secret;
    $url = "https://api.paystack.co{$endpoint}";
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    if ($post !== null) {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post));
    }
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer {$paystack_secret}",
        "Cache-Control: no-cache"
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    $resp = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);

    if ($err) {
        log_event("cURL error on {$endpoint}: {$err}");
        return ['status' => false, 'message' => 'Network error'];
    }
    $json = json_decode($resp, true);
    if (!$json) {
        log_event("Invalid JSON from {$endpoint}: {$resp}");
        return ['status' => false, 'message' => 'Invalid response'];
    }
    return $json;
}

// Refund helper: marks withdrawal as failed and refunds wallet
function refundWalletAndMarkFailed($con, $phone_number, $amount, $withdrawal_id) {
    // Use transaction to ensure both operations (refund + mark failed) happen together
    try {
        $con->begin_transaction();

        $stmt = $con->prepare("UPDATE members SET earning_wallet = earning_wallet + ? WHERE phone_number = ?");
        if (!$stmt) throw new Exception("Prepare refund1 failed: " . $con->error);
        $stmt->bind_param("ds", $amount, $phone_number);
        $stmt->execute();
        $stmt->close();

        $stmt = $con->prepare("UPDATE withdrawal_requests SET status='failed' WHERE id = ?");
        if (!$stmt) throw new Exception("Prepare refund2 failed: " . $con->error);
        $stmt->bind_param("i", $withdrawal_id);
        $stmt->execute();
        $stmt->close();

        $con->commit();
        log_event("Refund successful for withdraw_id={$withdrawal_id}, phone={$phone_number}, amount={$amount}");
    } catch (Throwable $e) {
        $con->rollback();
        log_event("Refund failed for withdraw_id={$withdrawal_id}, phone={$phone_number}, amount={$amount}. Error: " . $e->getMessage());
        // If refund couldn't be processed automatically, this needs manual intervention.
    }
}

// ----- AJAX endpoints ----- (unchanged)
if (isset($_GET['action']) && $_GET['action'] === 'bank_list') {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(paystack_call('/bank'));
    exit;
}
if (isset($_GET['action']) && $_GET['action'] === 'resolve') {
    header('Content-Type: application/json; charset=utf-8');
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['status' => false, 'message' => 'Use POST']);
        exit;
    }
    $csrf = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
    if (!$csrf || !hash_equals($_SESSION['csrf_token'] ?? '', $csrf)) {
        http_response_code(403);
        echo json_encode(['status' => false, 'message' => 'CSRF failed']);
        exit;
    }
    $account_number = trim($_POST['account_number'] ?? '');
    $bank_code = trim($_POST['bank_code'] ?? '');
    if ($account_number === '' || $bank_code === '') {
        http_response_code(400);
        echo json_encode(['status' => false, 'message' => 'Missing account_number/bank_code']);
        exit;
    }
    echo json_encode(paystack_call("/bank/resolve?account_number=" . urlencode($account_number) . "&bank_code=" . urlencode($bank_code)));
    exit;
}

// ----- Normal page logic -----
$phone_number = $_SESSION["phone_number"];
$message = "";
$message_type = "";

// Fetch user wallet
$stmt = $con->prepare("SELECT earning_wallet FROM members WHERE phone_number = ? LIMIT 1");
$stmt->bind_param("s", $phone_number);
$stmt->execute();
$stmt->bind_result($earning_wallet);
$stmt->fetch();
$stmt->close();
$earning_wallet = floatval($earning_wallet ?? 0.0); // ensure float

// Fetch PIN
$stmt = $con->prepare("SELECT pin FROM withdraw_pins WHERE phone_number = ? LIMIT 1");
$stmt->bind_param("s", $phone_number);
$stmt->execute();
$stmt->bind_result($stored_pin_hash);
$stmt->fetch();
$stmt->close();
$withdraw_pin_exists = !empty($stored_pin_hash);

// Redirect to set PIN if not set
if (!$withdraw_pin_exists) {
    header("Location: change_pin.php");
    exit;
}

// Rate-limit PIN attempts
$_SESSION['withdraw_pin_failures'] = $_SESSION['withdraw_pin_failures'] ?? 0;
$_SESSION['withdraw_pin_lock_until'] = $_SESSION['withdraw_pin_lock_until'] ?? 0;
$now = time();
if ($now < $_SESSION['withdraw_pin_lock_until']) {
    $remain = $_SESSION['withdraw_pin_lock_until'] - $now;
    $mins = ceil($remain / 60);
    $message = "Too many wrong PIN attempts. Try again in {$mins} minute(s).";
    $message_type = "error";
}

// Handle withdrawal - SAFER FLOW: deduct first, insert pending, commit, then call Paystack
if (empty($message) && isset($_POST['request_withdrawal'])) {
    $csrf = $_POST['csrf_token'] ?? '';
    if (!$csrf || !hash_equals($_SESSION['csrf_token'], $csrf)) {
        $message = "Security check failed. Please refresh and try again.";
        $message_type = "error";
    } else {
        // Gather input
        $amount = floatval($_POST['amount'] ?? 0); // force float
        $amount = round($amount, 2);
        $bank_code = $_POST['bank_code'] ?? '';
        $bank_name = $_POST['bank_name'] ?? '';
        $account_number = $_POST['account_number'] ?? '';
        $account_name_client = $_POST['account_name'] ?? '';
        $entered_pin = $_POST['withdraw_pin'] ?? '';

        // Limits
        $MIN_WITHDRAW = 100.00;
        $MAX_WITHDRAW = 10000000.00;

        // PIN check
        if (!password_verify($entered_pin, $stored_pin_hash)) {
            $_SESSION['withdraw_pin_failures']++;
            if ($_SESSION['withdraw_pin_failures'] >= 5) {
                $_SESSION['withdraw_pin_lock_until'] = time() + (10 * 60);
                $_SESSION['withdraw_pin_failures'] = 0;
            }
            $message = "Incorrect withdrawal PIN.";
            $message_type = "error";
        } elseif ($amount < $MIN_WITHDRAW || $amount > $MAX_WITHDRAW || $amount > $earning_wallet) {
            $message = "Invalid amount or insufficient balance.";
            $message_type = "error";
        } elseif (empty($bank_code) || empty($account_number) || empty($account_name_client)) {
            $message = "Please complete bank details and verify account name.";
            $message_type = "error";
        } else {
            // Verify account name with Paystack first (no money moved yet)
            $resolve = paystack_call("/bank/resolve?account_number=" . urlencode($account_number) . "&bank_code=" . urlencode($bank_code));
            if (!($resolve['status'] ?? false)) {
                $message = "Could not verify account at the moment. Please try again later.";
                $message_type = "error";
                log_event("Resolve failed: " . ($resolve['message'] ?? json_encode($resolve)));
            } else {
                $resolved_name = $resolve['data']['account_name'] ?? '';
                if (strcasecmp(trim($resolved_name), trim($account_name_client)) !== 0) {
                    $message = "Account name mismatch. Please re-check your details.";
                    $message_type = "error";
                } else {
                    // PREPARE: calculate charge and net
                    $charge = round($amount * 0.10, 2); // 10% charge
                    $net_amount = floatval(round($amount - $charge, 2));
                    if ($net_amount <= 0) {
                        $message = "Amount too small after charges.";
                        $message_type = "error";
                    } else {
                        // --------------- SAFER DB FLOW ---------------
                        // 1) Begin transaction, deduct wallet, insert withdrawal (status pending), commit
                        try {
                            $con->begin_transaction();

                            // Deduct from wallet
                            $stmt = $con->prepare("UPDATE members SET earning_wallet = earning_wallet - ? WHERE phone_number = ? AND earning_wallet >= ?");
                            if (!$stmt) {
                                throw new Exception("Prepare update wallet failed: " . $con->error);
                            }
                            // placeholders: amount (d), phone_number (s), amount (d)
                            $stmt->bind_param("dsd", $amount, $phone_number, $amount);
                            $stmt->execute();
                            $affected = $stmt->affected_rows;
                            $stmt->close();

                            if ($affected !== 1) {
                                // Not enough funds or phone mismatch
                                $con->rollback();
                                log_event("Wallet deduction failed for {$phone_number}. amount={$amount}. affected={$affected}");
                                $message = "Insufficient balance or wallet update failed. Please try again.";
                                $message_type = "error";
                            } else {
                                // Insert withdrawal request as PENDING (transaction_ref left NULL for now)
                                $stmt = $con->prepare("INSERT INTO withdrawal_requests (phone_number, bank_name, account_number, account_name, amount, status, request_date) VALUES (?, ?, ?, ?, ?, 'pending', NOW())");
                                if (!$stmt) {
                                    throw new Exception("Prepare insert pending failed: " . $con->error);
                                }
                                $stmt->bind_param("ssssd", $phone_number, $bank_name, $account_number, $resolved_name, $amount);
                                $stmt->execute();
                                $withdrawal_id = $stmt->insert_id;
                                $ok_insert = $stmt->affected_rows === 1;
                                $stmt->close();

                                if (!$ok_insert || $withdrawal_id <= 0) {
                                    $con->rollback();
                                    log_event("Insert pending withdrawal failed for {$phone_number}. amount={$amount}");
                                    $message = "Could not create withdrawal record. Please try again.";
                                    $message_type = "error";
                                } else {
                                    // Commit: wallet deducted and pending withdrawal created
                                    $con->commit();
                                    // Update local view of wallet
                                    $earning_wallet = round($earning_wallet - $amount, 2);
                                    log_event("Pending withdrawal created id={$withdrawal_id}, phone={$phone_number}, amount={$amount}");
                                    
                                    // 2) Now call Paystack (outside DB transaction)
                                    // Create transfer recipient
                                    $recipient_payload = [
                                        "type" => "nuban",
                                        "name" => $resolved_name,
                                        "account_number" => $account_number,
                                        "bank_code" => $bank_code,
                                        "currency" => "NGN"
                                    ];
                                    $recipient_res = paystack_call('/transferrecipient', $recipient_payload);
                                    if (!($recipient_res['status'] ?? false)) {
                                        // Paystack recipient creation failed -> refund and mark failed
                                        log_event("Recipient creation failed for withdraw_id={$withdrawal_id}, resp=" . json_encode($recipient_res));
                                        refundWalletAndMarkFailed($con, $phone_number, $amount, $withdrawal_id);
                                        $message = "Unable to process transfer at the moment. Your wallet has been refunded. Please try again later.";
                                        $message_type = "error";
                                    } else {
                                        $recipient_code = $recipient_res['data']['recipient_code'] ?? '';
                                        // Send transfer
                                        $transfer_payload = [
                                            "source" => "balance",
                                            "amount" => intval(round($net_amount * 100)), // kobo
                                            "recipient" => $recipient_code,
                                            "reason" => "Withdrawal from Affiliate Earnings"
                                        ];
                                        $transfer_res = paystack_call('/transfer', $transfer_payload);
                                        if (!($transfer_res['status'] ?? false)) {
                                            // Transfer failed -> refund and mark failed
                                            log_event("Transfer failed for withdraw_id={$withdrawal_id}, resp=" . json_encode($transfer_res));
                                            refundWalletAndMarkFailed($con, $phone_number, $amount, $withdrawal_id);
                                            $message = "Transfer failed. Your wallet has been refunded. Try again later.";
                                            $message_type = "error";
                                        } else {
                                            // Transfer succeeded -> update withdrawal record to completed with reference
                                            $transaction_ref = $transfer_res['data']['reference'] ?? '';
                                            try {
                                                $stmt = $con->prepare("UPDATE withdrawal_requests SET status='completed', transaction_ref = ? WHERE id = ?");
                                                if (!$stmt) {
                                                    // If we can't prepare, this is alarming: log and leave manual intervention note
                                                    log_event("Prepare update completed failed for id={$withdrawal_id}: " . $con->error);
                                                    // Attempt to mark as completed via another try (still do best effort)
                                                    throw new Exception("DB update prepare failed");
                                                }
                                                $stmt->bind_param("si", $transaction_ref, $withdrawal_id);
                                                $stmt->execute();
                                                $rows_upd = $stmt->affected_rows;
                                                $stmt->close();
                                                if ($rows_upd !== 1) {
                                                    // Not updated as expected - log critical
                                                    log_event("Warning: withdrawal_requests not updated to completed for id={$withdrawal_id}, rows_aff={$rows_upd}");
                                                    // But funds are already moved — show success but advise contacting support if they see discrepancy
                                                    $message = "Withdrawal processed. ₦" . number_format($amount, 2) . " deducted. ₦" . number_format($net_amount, 2) . " sent to your bank. Ref: {$transaction_ref}. If your dashboard doesn't reflect this, contact support with the Ref.";
                                                    $message_type = "success";
                                                } else {
                                                    $message = "Withdrawal successful. ₦" . number_format($amount, 2) . " deducted from wallet. ₦" . number_format($net_amount, 2) . " sent to your bank. Ref: {$transaction_ref}";
                                                    $message_type = "success";
                                                }
                                            } catch (Throwable $e) {
                                                log_event("Exception updating withdrawal_requests to completed for id={$withdrawal_id}: " . $e->getMessage());
                                                $message = "Withdrawal processed but we had an internal error updating the record. Please contact support with Ref: {$transaction_ref}";
                                                $message_type = "error";
                                            }
                                        } // end transfer result
                                    } // end recipient success
                                } // end insert ok
                            } // end affected === 1
                        } catch (Throwable $e) {
                            // If any exception inside DB steps, ensure rollback
                            try { $con->rollback(); } catch (Throwable $er) { /* ignore */ }
                            log_event("Exception during safer DB flow for phone={$phone_number}, amount={$amount}. Error: " . $e->getMessage());
                            $message = "A server error occurred while processing withdrawal. Please try again later.";
                            $message_type = "error";
                        } // end try-catch db flow
                        // --------------- END SAFER FLOW ---------------
                    } // end net_amount check
                } // end name match
            } // end resolve
        } // end validations
    } // end CSRF else
} // end handle withdrawal

// The rest of your page HTML/templating would follow here. Example usage of $message / $message_type.
// For debugging you can echo $message or render it in your UI as before.
?>




<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<title>Withdraw Funds</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
<script src="//cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<style>
 /* Global box-sizing for consistent sizing */
*, *::before, *::after {
  box-sizing: border-box;
}

body {
    margin: 0;
    font-family: 'Segoe UI', sans-serif;
    background: #1c1c1c; /* very dark gray/black background */
    color: #ddd; /* light gray text for readability */
}

header, footer {
    background: linear-gradient(135deg, black, #805c00); /* rich gold gradient */
    color: white; /* light text for contrast */
    text-align: center;
    padding: 20px 15px;
    font-weight: 700;
    letter-spacing: 1px;
    text-shadow: 0 0 5px rgba(0,0,0,0.3);
    box-shadow: 0 3px 6px rgba(0,0,0,0.5);
}

.container {
    max-width: 600px;
    margin: 40px auto;
    padding: 30px 20px; /* reduced horizontal padding */
    background: #2e2e2e; /* dark gray container */
    border-radius: 12px;
    box-shadow: 0 0 20px rgba(191,160,74,0.6); /* soft golden glow */
    color: #eee;
    overflow-x: hidden; /* prevent horizontal scroll */
}

h2 {
    text-align: center;
    color: #d4af37; /* gold color */
    
    margin-bottom: 25px;
}

/* Amount options container with responsive grid */
.amount-options {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(100px, 1fr)); /* flexible columns */
    gap: 10px;
}

/* 3 columns explicitly on small screens */
@media (max-width: 600px) {
    .amount-options {
        grid-template-columns: repeat(3, 1fr);
    }
}

.amount-options button {
    width: 100%;
    max-width: 100%;
    padding: 10px 10px;
    background: linear-gradient(145deg, black, #a67c00);
    color: white;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-weight: 600;
    box-shadow: 0 3px 6px rgba(166,124,0,0.6);
    transition: background 0.3s ease, transform 0.2s ease;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}
.amount-options button:hover {
    background: linear-gradient(145deg, #a67c00, #d4af37);
    transform: translateY(-2px);
    box-shadow: 0 6px 12px rgba(166,124,0,0.8);
}

input, select {
    width: 100%;
    padding: 14px;
    margin: 10px 0 24px;
    border-radius: 8px;
    border: 1.5px solid #bfa04a;
    background: #444; /* dark input background */
    color: #eee;
    font-weight: 600;
    box-shadow: inset 0 0 6px rgba(191,160,74,0.4);
    transition: border-color 0.3s ease;
}

input:focus, select:focus {
    border-color: #d4af37;
    outline: none;
    box-shadow: 0 0 8px #d4af37;
    background: #555;
    color: #fff;
}

button.submit-btn {
    width: 100%;
    padding: 16px;
    background: linear-gradient(135deg, black, #805c00);
    color: white;
    border: none;
    font-size: 18px;
    border-radius: 8px;
    cursor: pointer;
    font-weight: 700;
    box-shadow: 0 5px 15px rgba(191,160,74,0.7);
    transition: background 0.3s ease, transform 0.2s ease;
}

button.submit-btn:hover {
    background: linear-gradient(135deg, #805c00, #bfa04a);
    transform: translateY(-3px);
    box-shadow: 0 8px 20px rgba(191,160,74,0.9);
}

#verify_icon {
    font-size: 20px;
    vertical-align: middle;
    color: #d4af37;
    text-shadow: 0 0 4px #bfa04a;
}

#chargesInfo {
    background: #333; /* dark gray box */
    border: 1px solid #bfa04a;
    padding: 14px 18px;
    margin-bottom: 24px;
    border-radius: 8px;
    font-size: 16px;
    color: #d4af37;
    box-shadow: 0 0 10px rgba(191,160,74,0.6);
    font-weight: 600;
}

</style>

</head>
<body>
<header>
  <strong>Automatic Payout System</strong>
</header>

<div class="mobile-menu" id="mobileMenu">
  <a href="dashboard.php"><i class="fas fa-home"></i> Dashboard</a>
</div>

<div class="container">
     <h2><i class="fas fa-money-check-alt"></i> Withdraw Fund To Bank</h2>
     <p align="center"><strong>Earning Balance:</strong> ₦<?= number_format($earning_wallet, 2) ?></p>
     
     <br>
    <p><strong>Select Amount To Withdraw:</strong> </p>

    <form method="POST" id="withdrawForm" autocomplete="off">
        <!-- CSRF token -->
        <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($CSRF_TOKEN, ENT_QUOTES) ?>">

        <div class="amount-options">
            <?php foreach ([100,200,500,1000,2000,5000,10000,20000,45000,50000,100000,250000,500000,1000000] as $amt): ?>
                <button type="button" onclick="setAmount(<?= $amt ?>);">₦<?= number_format($amt) ?></button>
            <?php endforeach; ?>
        </div>

        <input type="number" name="amount" id="amount" min="100" placeholder="Amount" readonly required>

        <div id="chargesInfo">
            <div><strong>Transaction Charges (10%):</strong> ₦<span id="chargesAmount">0.00</span></div>
            <div><strong>Amount to Receive:</strong> ₦<span id="netAmount">0.00</span></div>
        </div>

        <label>Select Bank</label>
        <select id="bank" required>
            <option value="">Loading banks...</option>
        </select>

        <input type="hidden" name="bank_code" id="bank_code_input">
        <input type="hidden" name="bank_name" id="bank_name_input">

        <label>Account Number</label>
        <div style="display:flex;gap:10px;align-items:center;">
            <input type="text" name="account_number" id="account_number" placeholder="Enter Account Number" required maxlength="10" style="flex:1;">
            <span id="verify_icon"></span>
        </div>

        <label>Account Name</label>
        <input type="text" name="account_name" id="account_name" placeholder="Account Name" readonly required>

        <label>Withdrawal PIN</label>
        <input type="password" name="withdraw_pin" placeholder="Enter Withdrawal PIN" required maxlength="4" inputmode="numeric" pattern="\d{4}">

        <button class="submit-btn" type="submit" id="submitBtn" disabled name="request_withdrawal"><i class="fas fa-paper-plane"></i> Submit Request</button>
    </form>

    <?php if (!empty($message)): ?>
        <script>
            document.addEventListener("DOMContentLoaded", function(){
                Swal.fire({
                    icon: '<?= $message_type === "success" ? "success" : "error" ?>',
                    title: '<?= $message_type === "success" ? "Success" : "Error" ?>',
                    html: '<?= nl2br(htmlspecialchars($message, ENT_QUOTES)) ?>'
                }).then(() => {
                    <?php if ($message_type === 'success'): ?>
                        window.location.href = 'withdrawal_history.php';
                    <?php endif; ?>
                });
            });
        </script>
    <?php endif; ?>

    <p style="margin-top: 20px; text-align:center;">
        Need to set or reset your withdrawal PIN? <a href="change_pin.php" style="color:#004080; font-weight:bold;">Click here</a>.
    </p>
</div>

<script>
const proxy = '<?= basename(__FILE__) ?>'; // this same file handles AJAX endpoints
const CSRF = <?= json_encode($CSRF_TOKEN) ?>; // reuse CSRF for AJAX
const bankSelect = document.getElementById('bank');
const bankCodeInput = document.getElementById('bank_code_input');
const bankNameInput = document.getElementById('bank_name_input');
const accountInput = document.getElementById('account_number');
const accountName = document.getElementById('account_name');
const verifyIcon = document.getElementById('verify_icon');
const submitBtn = document.getElementById('submitBtn');
const amountInput = document.getElementById('amount');

function setAmount(val) {
    amountInput.value = val;
    updateCharges();
}

function updateCharges() {
    let amount = parseFloat(amountInput.value) || 0;
    if (amount < 100) {
        document.getElementById('chargesAmount').textContent = '0.00';
        document.getElementById('netAmount').textContent = '0.00';
        submitBtn.disabled = true;
        return;
    }
    let charge = amount * 0.10;
    let payout = amount - charge;
    document.getElementById('chargesAmount').textContent = charge.toFixed(2);
    document.getElementById('netAmount').textContent = payout.toFixed(2);
    submitBtn.disabled = !(verified && amount >= 100);
}

document.querySelectorAll('.amount-options button').forEach(btn => {
    btn.addEventListener('click', function() {
        let val = parseFloat(this.textContent.replace(/[^0-9]/g, ''));
        setAmount(val);
    });
});

amountInput.addEventListener('input', updateCharges);

// Load bank list from server (server calls Paystack with secret key)
fetch(proxy + '?action=bank_list')
.then(r => r.json())
.then(data => {
    bankSelect.innerHTML = '<option value="">-- Select Bank --</option>';
    if (data.status && Array.isArray(data.data)) {
        data.data.forEach(b => {
            const opt = document.createElement('option');
            opt.value = b.code + '||' + b.name;
            opt.textContent = b.name;
            bankSelect.appendChild(opt);
        });
    } else {
        bankSelect.innerHTML = '<option value="">Unable to load banks</option>';
    }
})
.catch(() => {
    bankSelect.innerHTML = '<option value="">Error loading banks</option>';
});

let verified = false;
function setVerified(ok, name='') {
    verified = ok;
    if (ok) {
        verifyIcon.innerHTML = '✅';
        verifyIcon.style.color = 'green';
        accountName.value = name;
        updateCharges();
    } else {
        verifyIcon.innerHTML = '❌';
        verifyIcon.style.color = 'red';
        accountName.value = '';
        submitBtn.disabled = true;
    }
}

function clearVerify() {
    verified = false;
    verifyIcon.innerHTML = '';
    accountName.value = '';
    submitBtn.disabled = true;
    updateCharges();
}

function resolveAccount() {
    const sel = bankSelect.value;
    if (!sel) { clearVerify(); return; }
    const [bank_code, bank_name] = sel.split('||');
    const acc = accountInput.value.replace(/\D/g,'').trim();
    if (acc.length !== 10) { clearVerify(); return; }

    verifyIcon.innerHTML = '⏳';
    verifyIcon.style.color = 'blue';
    accountName.value = '';

    const form = new FormData();
    form.append('account_number', acc);
    form.append('bank_code', bank_code);

    fetch(proxy + '?action=resolve', {
        method: 'POST',
        headers: { 'X-CSRF-Token': CSRF },
        body: form
    })
    .then(r => r.json())
    .then(data => {
        if (data.status && data.data && data.data.account_name) {
            setVerified(true, data.data.account_name);
            bankCodeInput.value = bank_code;
            bankNameInput.value = bank_name;
        } else {
            setVerified(false);
        }
    })
    .catch(() => {
        verifyIcon.innerHTML = '⚠️';
        verifyIcon.style.color = 'orange';
        accountName.value = 'Error verifying';
        submitBtn.disabled = true;
        verified = false;
    });
}

bankSelect.addEventListener('change', resolveAccount);
accountInput.addEventListener('input', () => {
    // delay verify until 10 digits entered
    if (accountInput.value.replace(/\D/g,'').length === 10) {
        resolveAccount();
    } else {
        clearVerify();
    }
});
</script>

<footer>
    <p>&copy; <?= date('Y') ?> <?= htmlspecialchars($_SERVER['HTTP_HOST']) ?>. All rights reserved.</p>
</footer>
</body>
</html>
