<?php
// paystack-webhook.php
// Your webhook endpoint must be reachable over HTTPS and set in Paystack Dashboard.

header('Content-Type: application/json');

require_once __DIR__ . '/env-helper.php';
require_once __DIR__ . '/connect.php';

// --- Config ---
$paystack_secret = getEnvValue('PAYSTACK_DEPOSIT_SECRET');
if (!$paystack_secret) {
    error_log("[WEBHOOK] Missing PAYSTACK_DEPOSIT_SECRET in .env");
    http_response_code(500);
    exit(json_encode(['status' => 'error', 'message' => 'Server not configured']));
}

// Optional: IP allowlist (Paystack IPs). Set to true to enforce.
$ENFORCE_IP_WHITELIST = false;
$ALLOWED_IPS = ['52.31.139.75','52.49.173.169','52.214.14.220'];

// --- Helpers ---
function extract_phone_from_metadata(array $meta): ?string {
    // 1) Direct key
    if (!empty($meta['phone_number'])) {
        return trim($meta['phone_number']);
    }
    // 2) custom_fields (legacy style)
    if (!empty($meta['custom_fields']) && is_array($meta['custom_fields'])) {
        foreach ($meta['custom_fields'] as $field) {
            $var = strtolower($field['variable_name'] ?? '');
            if ($var === 'phone_number' || $var === 'phone' || $var === 'msisdn') {
                return trim($field['value'] ?? '');
            }
        }
    }
    return null;
}

function amount_to_string($n)
{
    // Store into your VARCHAR amount column as a clean decimal string
    return number_format((float)$n, 2, '.', '');
}

// --- Read & verify request ---
$client_ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
if ($ENFORCE_IP_WHITELIST && !in_array($client_ip, $ALLOWED_IPS, true)) {
    error_log("[WEBHOOK] Blocked by IP whitelist: {$client_ip}");
    http_response_code(403);
    exit(json_encode(['status' => 'error', 'message' => 'Forbidden']));
}

$raw = @file_get_contents('php://input');
if (!$raw) {
    error_log("[WEBHOOK] Empty body");
    http_response_code(400);
    exit(json_encode(['status' => 'error', 'message' => 'Empty body']));
}
error_log("[WEBHOOK] Raw: " . $raw);

$signature = $_SERVER['HTTP_X_PAYSTACK_SIGNATURE'] ?? '';
if (!$signature) {
    error_log("[WEBHOOK] Missing signature header");
    http_response_code(400);
    exit(json_encode(['status' => 'error', 'message' => 'Missing signature']));
}

$computed = hash_hmac('sha512', $raw, $paystack_secret);
if (!hash_equals($signature, $computed)) {
    error_log("[WEBHOOK] Invalid signature (IP: {$client_ip})");
    http_response_code(400);
    exit(json_encode(['status' => 'error', 'message' => 'Invalid signature']));
}

$payload = json_decode($raw, true);
if (!is_array($payload) || empty($payload['event']) || empty($payload['data'])) {
    error_log("[WEBHOOK] Invalid JSON payload");
    http_response_code(400);
    exit(json_encode(['status' => 'error', 'message' => 'Invalid payload']));
}

if ($payload['event'] !== 'charge.success') {
    error_log("[WEBHOOK] Ignored event: {$payload['event']}");
    http_response_code(200);
    exit(json_encode(['status' => 'ignored']));
}

// --- Extract fields ---
$data         = $payload['data'];
$reference    = $data['reference'] ?? '';
$status       = $data['status']    ?? '';
$amount_ngn   = isset($data['amount']) ? ($data['amount'] / 100.0) : 0.0;
$metadata     = is_array($data['metadata'] ?? null) ? $data['metadata'] : [];
$customerMail = $data['customer']['email'] ?? '';

if (!$reference || $status !== 'success' || $amount_ngn <= 0) {
    error_log("[WEBHOOK] Missing/invalid core fields: ref={$reference}, status={$status}, amount={$amount_ngn}");
    http_response_code(400);
    exit(json_encode(['status' => 'error', 'message' => 'Missing required fields']));
}

// Resolve phone_number
$phone_number = extract_phone_from_metadata($metadata);

// Fallback 1: match member by the reference we saved at initialize
if (!$phone_number) {
    $stmt = $con->prepare("SELECT phone_number FROM members WHERE payment_reference = ? LIMIT 1");
    $stmt->bind_param("s", $reference);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($row = $res->fetch_assoc()) {
        $phone_number = $row['phone_number'];
        error_log("[WEBHOOK] Phone resolved via members.payment_reference => {$phone_number}");
    }
    $stmt->close();
}

// Fallback 2: match by email_address if available
if (!$phone_number && $customerMail) {
    $stmt = $con->prepare("SELECT phone_number FROM members WHERE email_address = ? LIMIT 1");
    $stmt->bind_param("s", $customerMail);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($row = $res->fetch_assoc()) {
        $phone_number = $row['phone_number'];
        error_log("[WEBHOOK] Phone resolved via members.email_address => {$phone_number}");
    }
    $stmt->close();
}

if (!$phone_number) {
    // We don’t know who to credit; still store a deposit row for reconciliation
    error_log("[WEBHOOK] Could not resolve phone_number for ref={$reference}");
}

// --- Begin transaction ---
$con->begin_transaction();

try {
    // Check existing deposit for this reference
    $stmt = $con->prepare("SELECT id, status FROM deposits WHERE payment_reference = ? LIMIT 1");
    $stmt->bind_param("s", $reference);
    $stmt->execute();
    $dep = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    $amount_str = amount_to_string($amount_ngn);

    if ($dep) {
        if ($dep['status'] === 'approved') {
            // Idempotent exit
            $con->commit();
            error_log("[WEBHOOK] Already approved: ref={$reference}");
            http_response_code(200);
            exit(json_encode(['status' => 'ok', 'message' => 'Already processed']));
        }
        // Update existing deposit
        $stmt = $con->prepare("
            UPDATE deposits 
               SET amount = ?, status = 'approved', date_submitted = NOW()
             WHERE id = ?
        ");
        $stmt->bind_param("si", $amount_str, $dep['id']);
        $stmt->execute();
        $stmt->close();
    } else {
        // Insert new deposit row (even if phone_number couldn't be resolved)
        $depositor = $phone_number ?: ($customerMail ?: 'unknown');
        $stmt = $con->prepare("
            INSERT INTO deposits (phone_number, depositor, amount, proof, date_submitted, status, payment_reference)
            VALUES (?, ?, ?, '', NOW(), 'approved', ?)
        ");
        $pn = $phone_number ?: ''; // your schema allows empty string
        $stmt->bind_param("ssss", $pn, $depositor, $amount_str, $reference);
        $stmt->execute();
        $stmt->close();
    }

    // Credit wallet once (only if we have a phone_number)
    if ($phone_number) {
        // members.main_wallet is VARCHAR; add safely using CAST then store back as string
        $stmt = $con->prepare("
            UPDATE members 
               SET main_wallet = (
                     CAST(IFNULL(NULLIF(main_wallet,''), '0') AS DECIMAL(20,2)) + ?
               )
             WHERE phone_number = ?
             LIMIT 1
        ");
        $stmt->bind_param("ds", $amount_ngn, $phone_number);
        $stmt->execute();
        $affected = $stmt->affected_rows;
        $stmt->close();

        if ($affected < 1) {
            throw new Exception("Member not found or wallet not updated for phone {$phone_number}");
        }

        // Optional: clear temp fields (your columns are NOT NULL, so set to empty string)
        $stmt = $con->prepare("
            UPDATE members 
               SET payment_reference = '', attempt_payment_amount = ''
             WHERE phone_number = ?
             LIMIT 1
        ");
        $stmt->bind_param("s", $phone_number);
        $stmt->execute();
        $stmt->close();
    }

    $con->commit();

    error_log("[WEBHOOK] SUCCESS: ref={$reference} phone={$phone_number} amount={$amount_str}");
    http_response_code(200);
    echo json_encode(['status' => 'success']);
} catch (Throwable $e) {
    $con->rollback();
    error_log("[WEBHOOK] ERROR: " . $e->getMessage() . " | ref={$reference}");
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Internal Server Error']);
}
